/*****************************************************************************
 * main.c 
 *
 * Project: SCT Application Example for LPC1800
 *
 * Description:
 *   Implements a special purpose Brushless DC example application
 *
 *----------------------------------------------------------------------------
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.
 *****************************************************************************/

/* For this application the following interface pins are used:
 * ----------------------------------------------------------------------------------
 * 1) NXP validation board:
 *
 *
 * Signal    Port   Connector           Signal
 * -----------------------------------------------------------------------------------  
 * CTIN_0    P2_4   I2C_1, pin 6        ABORT
 * CTIN_1    P2_3   I2C_1, pin 7		H1
 * CTIN_3    P7_3   LCD_UNI, pin 22     H2
 * CTIN_4    P7_2   LCD_UNI, pin 23		H3
 * CTIN_6    P4_9   LCD_UNI, pin 19		DEBUG
 * CTOUT_0   P4_2   LCD_UNI, pin 9      SW1
 * CTOUT_1   P4_1   LCD_UNI, pin 6      SW2
 * CTOUT_2   P4_4   LCD_UNI, pin 7     	SW3
 * CTOUT_3   P4_3   LCD_UNI, pin 8      SW4
 * CTOUT_4   P4_6   LCD_UNI, pin 35     SW5
 * CTOUT_5   P4_5   LCD_UNI, pin 28   	SW6
 * CTOUT_8   P7_7   LCD_UNI, pin 34     DEAD
 * CTOUT_11  P7_6   LCD_UNI, pin 30   	TRIGGER
 *
 * -----------------------------------------------------------------------------------
 *
 *
 * 2) NXP evaluation board v4.1:
 *
 * Signal    Port   Connector           Signal
 * ----------------------------------------------------------------------------------- 
 * Not enough pins available
 *            
 * -----------------------------------------------------------------------------------
 *
 *
 * 3) Hitex board:  
 *
 * Signal    Port   Connector           Signal
 * -----------------------------------------------------------------------------------   
 * CTIN_0    P2_4   X3, pin 14			ABORT
 * CTIN_1    P2_3   X3, pin 11			H1
 * CTIN_3    P7_3   SV18, pin 16 		H2
 * CTIN_4    P7_2   X11, pin 21			H3
 * CTIN_6    P4_9   X11, pin 14			DEBUG
 * CTOUT_0   P4_2   X11, pin 7    		SW1
 * CTOUT_1   P4_1   JP20, pin 2  		SW2
 * CTOUT_2   P4_4   JP22, pin 2  		SW3
 * CTOUT_3   P4_3   X11, pin 6    		SW4
 * CTOUT_4   P4_6   X11, pin 35   		SW5
 * CTOUT_5   P4_5   X11, pin 28  		SW6
 * CTOUT_8   P7_7   X11, pin 34     	DEAD
 * CTOUT_11  P7_6   X11, pin 30   		TRIGGER
 *          
 * -----------------------------------------------------------------------------------
 */


#include <stdio.h>

#ifdef LPC43XX
    #include "LPC43xx_TC2.h"
    #include "scu_LPC43xx.h"
    #include "emc_LPC43xx.h"
#endif

#ifdef LPC18XX
    #include "LPC18xx.h"
    #include "scu_LPC18xx.h"
    #include "emc_LPC18xx.h"
#endif


#include "sct_fsm.h"




// Switch positions
// Entry state: match1, sw1 & sw4, H123 = 100
// sect1: match2, sw1 & sw6, H123 = 110
// sect2: match3, sw6 & sw3, H123 = 010
// sect3: match4, sw3 & sw2, H123 = 011
// sect4: match5, sw2 & sw5, H123 = 001
// sect5: match6, sw5 & sw4, H123 = 101
unsigned char position_table[6] = { 3, 6, 4, 2, 1, 5 };


// state names assignment:
//  State L 0: ERROR
//  State L 1: sect4
//  State L 2: sect3
//  State L 3: L_ENTRY
//  State L 4: sect2
//  State L 5: sect5 
//  State L 6: sect1 
  

typedef struct sensor_input_tag 
{
    unsigned char H1;
    unsigned char H2;
    unsigned char H3;
} sensor_inputs;

  
sensor_inputs hall_config[6] = { {1,0,0}, {1,1,0}, {0,1,0}, {0,1,1}, {0,0,1}, {1,0,1} };



/******************************************************************************
 * Initialize board type specific IO's
 *  - NXP validation board
 *  - NXP evaluation board v4.1
 *  - Hitex board version A2
 ******************************************************************************/
void IOInit(void)
{

#ifdef NXP_VALIDATION_BOARD 
    // Board (and chip) specific initializations - NXP validation board 
    #ifdef LPC43XX 
        scu_pinmux(9, 2, MD_PDN, FUNC2);    // GPIO4_14: LD11   
        scu_pinmux(0xA ,4 , MD_PDN, FUNC2); // GPIO4_11: LD10
        LPC_GPIO4->DIR |= (1<<14);          // GPIO4_11 = output
        LPC_GPIO4->DIR |= (1<<11);          // GPIO4_11 = output

        scu_pinmux(6 ,10, MD_PUP, FUNC3);   // GPIO3_6: SW4 (button 0)
        scu_pinmux(4 , 0, MD_PUP, FUNC0);   // GPIO2_0: SW3 (button 1)
        LPC_GPIO3->DIR &= ~(1<<6);          // Button 0 = input
        LPC_GPIO2->DIR &= ~(1<<0);          // Button 1 = input
    
        // Pin configuration for SCT inputs/outputs - NXP validation board 
        scu_pinmux (4, 2, MD_PUP, FUNC1);   // P4_2:  SCTOUT_0 - LCD_UNI, pin 9 
        scu_pinmux (4, 1, MD_PLN, FUNC1);   // P4_1:  SCTOUT_1 - LCD_UNI, pin 6
        scu_pinmux (4, 4, MD_PLN, FUNC1);   // P4_4:  SCTOUT_2 - LCD_UNI, pin 7
        scu_pinmux (4, 3, MD_PLN, FUNC1);   // P4_3:  SCTOUT_3 - LCD_UNI, pin 8
        scu_pinmux (4, 6, MD_PLN, FUNC1);   // P4_6:  SCTOUT_4 - LCD_UNI, pin 35
        scu_pinmux (4, 5, MD_PLN, FUNC1);   // P4_5:  SCTOUT_5 - LCD_UNI, pin 28
		scu_pinmux (7, 7, MD_PLN, FUNC1);   // P7_7:  SCTOUT_8 - LCD_UNI, pin 34
        scu_pinmux (7, 6, MD_PLN, FUNC1);   // P7_6:  SCTOUT_11 - LCD_UNI, pin 30
    
        scu_pinmux (2, 4, MD_PDN, FUNC0);   // P2_4:  SCTIN_0 - I2C_1, pin 6
		scu_pinmux (2, 3, MD_PDN, FUNC0);   // P2_3:  SCTIN_1 - I2C_1, pin 7
        // SCTIN_2 does not work on the LPC4300 test chip
        //scu_pinmux (4,10, MD_PDN, FUNC1);   // P4_10: SCTIN_2 - LCD_UNI, pin 16 
        scu_pinmux (7, 3, MD_PDN, FUNC1);   // P7_3:  SCTIN_3 - LCD_UNI, pin 22 
		scu_pinmux (7, 2, MD_PDN, FUNC1);   // P7_3:  SCTIN_4 - LCD_UNI, pin 23 
		scu_pinmux (4, 9, MD_PDN, FUNC1);   // P7_3:  SCTIN_6 - LCD_UNI, pin 19 
    #endif

    #ifdef LPC18XX 
        scu_pinmux(9, 2, MD_PDN, FUNC0);    // GPIO4_14: LD11

        // on LPC1800 port pin PA_4 is not available    
        //scu_pinmux(0xA ,4 , MD_PDN, FUNC2); // GPIO4_11: LD10
        //LPC_GPIO4->DIR |= (1<<14);            // GPIO4_11 = output

        scu_pinmux(6 ,10, MD_PUP, FUNC3);   // GPIO3_6: SW4 (button 0)
        scu_pinmux(4 , 0, MD_PUP, FUNC0);   // GPIO2_0: SW3 (button 1)
    
        LPC_GPIO4->DIR |= (1<<14);          // LD11 = output
        LPC_GPIO3->DIR &= ~(1<<6);          // Button 0 = input
        LPC_GPIO2->DIR &= ~(1<<0);          // Button 1 = input
    
        // Pin configuration for SCT inputs/outputs - NXP validation board 
        scu_pinmux (4, 2, MD_PUP, FUNC1);   // P4_2:  SCTOUT_0 - LCD_UNI, pin 9 
        scu_pinmux (4, 1, MD_PLN, FUNC1);   // P4_1:  SCTOUT_1 - LCD_UNI, pin 6
        scu_pinmux (4, 4, MD_PLN, FUNC1);   // P4_4:  SCTOUT_2 - LCD_UNI, pin 7
        scu_pinmux (4, 3, MD_PLN, FUNC1);   // P4_3:  SCTOUT_3 - LCD_UNI, pin 8
        scu_pinmux (4, 6, MD_PLN, FUNC1);   // P4_6:  SCTOUT_4 - LCD_UNI, pin 35
        scu_pinmux (4, 5, MD_PLN, FUNC1);   // P4_5:  SCTOUT_5 - LCD_UNI, pin 28
		scu_pinmux (7, 7, MD_PLN, FUNC1);   // P7_7:  SCTOUT_8 - LCD_UNI, pin 34
        scu_pinmux (7, 6, MD_PLN, FUNC1);   // P7_6:  SCTOUT_11 - LCD_UNI, pin 30
    
        scu_pinmux (2, 4, MD_PDN, FUNC0);   // P2_4:  SCTIN_0 - I2C_1, pin 6
		scu_pinmux (2, 3, MD_PDN, FUNC3);   // P2_3:  SCTIN_1 - I2C_1, pin 7
        scu_pinmux (4,10, MD_PDN, FUNC1);   // P4_10: SCTIN_2 - LCD_UNI, pin 16 
        scu_pinmux (7, 3, MD_PDN, FUNC1);   // P7_3:  SCTIN_3 - LCD_UNI, pin 22
		scu_pinmux (7, 2, MD_PDN, FUNC1);   // P7_3:  SCTIN_4 - LCD_UNI, pin 23 
		scu_pinmux (4, 9, MD_PDN, FUNC1);   // P7_3:  SCTIN_6 - LCD_UNI, pin 19  
    #endif
#endif


#ifdef NXP_EVAL_BOARD   // Not enough ports available on this board !!! 
    // Board (and chip) specific initializations - NXP eval board v4.1
	/*
    #ifdef LPC43XX
        scu_pinmux(8, 0, MD_PLN, FUNC3);    // GPIO4_0: on-board LED0   
        scu_pinmux(8 ,1, MD_PLN, FUNC3);    // GPIO4_1: on-board LED1
        scu_pinmux(6,10, MD_PUP, FUNC3);    // GPIO3_6: on board SW2 (button 0)
        scu_pinmux(4, 0, MD_PUP, FUNC0);    // GPIO2_0: on-board SW3 (button 1)
    
        LPC_GPIO4->DIR |= 0x0003;           // LED0 and LED1 = output
        LPC_GPIO3->DIR &= ~(1<<6);          // Button 0 = input
        LPC_GPIO2->DIR &= ~(1<<0);          // Button 1 = input
    
        // Pin configuration for SCT inputs/outputs - NXP eval board v4.1
        scu_pinmux (4, 2, MD_PUP, FUNC1);   // P4_2:  SCTOUT_0 - X1, pin 3
        scu_pinmux (4, 1, MD_PLN, FUNC1);   // P4_1:  SCTOUT_1 - X1, pin 4
        scu_pinmux (4, 4, MD_PLN, FUNC1);   // P4_4:  SCTOUT_2 - X1, pin 5
        scu_pinmux (4, 3, MD_PLN, FUNC1);   // P4_3:  SCTOUT_3 - X1, pin 6
        scu_pinmux (4, 6, MD_PLN, FUNC1);   // P4_6:  SCTOUT_4 - X1, pin 7
        scu_pinmux (4, 5, MD_PLN, FUNC1);   // P4_5:  SCTOUT_5 - X1, pin 8
    
        scu_pinmux (0xD, 10, MD_PDN, FUNC1);    // PD_10: SCTIN_1 - X1, pin 20
        // SCTIN_2 does not work on the LPC4300 test chip
        //scu_pinmux (2, 5,    MD_PDN, FUNC1);      // P2_5:  SCTIN_2 - JP7, pin 1
        scu_pinmux (7, 3,    MD_PDN, FUNC1);    // P7_3:  SCTIN_3 - X1, pin 22
    #endif

    #ifdef LPC18XX
        scu_pinmux(8, 0, MD_PLN, FUNC0);    // GPIO4_0: on-board LED0   
        scu_pinmux(8 ,1, MD_PLN, FUNC0);    // GPIO4_1: on-board LED1
        scu_pinmux(6,10, MD_PUP, FUNC0);    // GPIO3_6: on board SW2 (button 0)
        scu_pinmux(4, 0, MD_PUP, FUNC0);    // GPIO2_0: on-board SW3 (button 1)
    
        LPC_GPIO4->DIR |= 0x0003;           // LED0 and LED1 = output
        LPC_GPIO3->DIR &= ~(1<<6);          // Button 0 = input
        LPC_GPIO2->DIR &= ~(1<<0);          // Button 1 = input
    
        // Pin configuration for SCT inputs/outputs - NXP eval board v4.1
        scu_pinmux (4, 2, MD_PUP, FUNC1);   // P4_2:  SCTOUT_0 - X1, pin 3
        scu_pinmux (4, 1, MD_PLN, FUNC1);   // P4_1:  SCTOUT_1 - X1, pin 4
        scu_pinmux (4, 4, MD_PLN, FUNC1);   // P4_4:  SCTOUT_2 - X1, pin 5
        scu_pinmux (4, 3, MD_PLN, FUNC1);   // P4_3:  SCTOUT_3 - X1, pin 6
        scu_pinmux (4, 6, MD_PLN, FUNC1);   // P4_6:  SCTOUT_4 - X1, pin 7
        scu_pinmux (4, 5, MD_PLN, FUNC1);   // P4_5:  SCTOUT_5 - X1, pin 8
    
        scu_pinmux (0xD, 10, MD_PDN, FUNC1);    // PD_10: SCTIN_1 - X1, pin 20
        scu_pinmux (2, 5,    MD_PDN, FUNC1);    // P2_5:  SCTIN_2 - JP7, pin 1
        scu_pinmux (7, 3,    MD_PDN, FUNC1);    // P7_3:  SCTIN_3 - X1, pin 22
    #endif
	*/
#endif

#ifdef HITEX_BOARD
    // Board (and chip) specific initializations - Hitex board
    #ifdef LPC43XX
        scu_pinmux(1 ,15 , MD_PLN, FUNC2);  // GPIO0_2: SV6, pin 11 used for external LED0
        scu_pinmux(1 ,16 , MD_PLN, FUNC2);  // GPIO0_3: SV6, pin 5 used for external LED1
        scu_pinmux(1 ,18 , MD_PUP, FUNC3);  // GPIO0_13: SV3, pin 11 used for external button 0
        scu_pinmux(1 ,20 , MD_PUP, FUNC3);  // GPIO0_15: SV3, pin 9 used for external button 1

        LPC_GPIO0->DIR |= 0x000A;           // LED0 and LED1 = output
        LPC_GPIO0->DIR &= ~(0xA000);        // Button 0 and Button 1 = input
        
        // Pin configuration for SCT - Hitex board
        scu_pinmux(4, 2, MD_PUP, FUNC1);   // P4_2:  SCTOUT_0 - X11, pin 7
        scu_pinmux(4, 1, MD_PLN, FUNC1);   // P4_1:  SCTOUT_1 - JP20, pin 2
        scu_pinmux(4, 4, MD_PLN, FUNC1);   // P4_4:  SCTOUT_2 - JP22, pin 2
        scu_pinmux(4, 3, MD_PLN, FUNC1);   // P4_3:  SCTOUT_3 - X11, pin 6
        scu_pinmux(4, 6, MD_PLN, FUNC1);   // P4_6:  SCTOUT_4 - X11, pin 35
        scu_pinmux(4, 5, MD_PLN, FUNC1);   // P4_5:  SCTOUT_5 - X11, pin 28
        scu_pinmux(7, 7, MD_PLN, FUNC1);   // P7_7:  SCTOUT_8 - X11, pin 34
        scu_pinmux(0xB, 0, MD_PLN, FUNC1); // PB_0:  SCTOUT_10 - X11, pin 27
        scu_pinmux(7, 6, MD_PLN, FUNC1);   // P7_6:  SCTOUT_11 - X11, pin 30
        scu_pinmux(7, 5, MD_PLN, FUNC1);   // P7_5:  SCTOUT_12 - SV18, pin 2
        scu_pinmux(7, 0, MD_PLN, FUNC1);   // P7_0:  SCTOUT_14 - JP19, pin 1
        scu_pinmux(7, 1, MD_PLN, FUNC1);   // P7_1:  SCTOUT_15 - X11, pin 22
    
        scu_pinmux(2, 4,  MD_PUP, FUNC0);  // P2_4:  SCTIN_0 - X3, pin 14
        scu_pinmux(2, 3,  MD_PUP, FUNC0);  // P2_3:  SCTIN_1 - X3, pin 11
        // SCTIN_2 does not work on the LPC4300 test chip
        //scu_pinmux(4, 10, MD_PUP, FUNC1);  // P4_10: SCTIN_2 - X11, pin 13   // does not work
        //scu_pinmux(0xF, 8, MD_PUP, FUNC0);  // P4_10: SCTIN_2 - X5, pin 20   // does not work
        //scu_pinmux(2, 5,  MD_PUP, FUNC1);  // P2_5: SCTIN_2 - IC U13, pin 8   // does not work
        scu_pinmux(7, 3,  MD_PUP, FUNC1);  // P7_3:  SCTIN_3 - SV18, pin 16
        scu_pinmux(7, 2,  MD_PUP, FUNC1);  // P7_2:  SCTIN_4 - X11, pin 21
        scu_pinmux(4, 9,  MD_PUP, FUNC1);  // P4_9:  SCTIN_6 - X11, pin 11
    #endif
    
    #ifdef LPC18XX
        scu_pinmux(1 ,15 , MD_PLN, FUNC0);  // GPIO0_2: SV6, pin 11 used for external LED0
        scu_pinmux(1 ,16 , MD_PLN, FUNC0);  // GPIO0_3: SV6, pin 5 used for external LED1
        scu_pinmux(1 ,18 , MD_PUP, FUNC0);  // GPIO0_13: SV3, pin 11 used for external button 0
        scu_pinmux(1 ,20 , MD_PUP, FUNC0);  // GPIO0_15: SV3, pin 9 used for external button 1

        LPC_GPIO0->DIR |= 0x000A;           // LED0 and LED1 = output
        LPC_GPIO0->DIR &= ~(0xA000);        // Button 0 and Button 1 = input
        
        // Pin configuration for SCT - Hitex board
        scu_pinmux(4, 2, MD_PUP, FUNC1);   // P4_2:  SCTOUT_0 - X11, pin 7
        scu_pinmux(4, 1, MD_PLN, FUNC1);   // P4_1:  SCTOUT_1 - JP20, pin 2
        scu_pinmux(4, 4, MD_PLN, FUNC1);   // P4_4:  SCTOUT_2 - JP22, pin 2
        scu_pinmux(4, 3, MD_PLN, FUNC1);   // P4_3:  SCTOUT_3 - X11, pin 6
        scu_pinmux(4, 6, MD_PLN, FUNC1);   // P4_6:  SCTOUT_4 - X11, pin 35
        scu_pinmux(4, 5, MD_PLN, FUNC1);   // P4_5:  SCTOUT_5 - X11, pin 28
		scu_pinmux(7, 7, MD_PLN, FUNC1);   // P7_7:  SCTOUT_8 - X11, pin 34
        scu_pinmux(0xB, 0, MD_PLN, FUNC1); // PB_0:  SCTOUT_10 - X11, pin 27
        scu_pinmux(7, 6, MD_PLN, FUNC1);   // P7_6:  SCTOUT_11 - X11, pin 30
        scu_pinmux(7, 5, MD_PLN, FUNC1);   // P7_5:  SCTOUT_12 - SV18, pin 2
        scu_pinmux(7, 0, MD_PLN, FUNC1);   // P7_0:  SCTOUT_14 - JP19, pin 1
        scu_pinmux(7, 1, MD_PLN, FUNC1);   // P7_1:  SCTOUT_15 - X11, pin 22
    
        scu_pinmux(2, 4,  MD_PUP, FUNC3);  // P2_4:  SCTIN_0 - X3, pin 14
        scu_pinmux(2, 3,  MD_PUP, FUNC3);  // P2_3:  SCTIN_1 - X3, pin 11
        scu_pinmux(4, 10, MD_PUP, FUNC1);  // P4_10: SCTIN_2 - X11, pin 13
        scu_pinmux(7, 3,  MD_PDN, FUNC1);  // P7_3:  SCTIN_3 - SV18, pin 16
        scu_pinmux(7, 2,  MD_PDN, FUNC1);  // P7_2:  SCTIN_4 - X11, pin 21
        scu_pinmux(4, 9,  MD_PUP, FUNC1);  // P4_9:  SCTIN_6 - X11, pin 11
    #endif
#endif
}



/******************************************************************************
 * Initialize special board specific IO
 * 
 *
 *  
 ******************************************************************************/
void SpecialIOInit(void)
{

    // Used for external loopback for simulating hall sensor H1
    LPC_GPIO4->DIR |= (1<<5);   // GPIO4[5] on P8_5 - Hitex board: ULPI pin 4

    // Used for external loopback for simulating hall sensor H2
    LPC_GPIO4->DIR |= (1<<4);   // GPIO4[4] on P8_4 - Hitex board: ULPI pin 5

    // Used for external loopback for simulating hall sensor H3
    LPC_GPIO4->DIR |= (1<<7);   // GPIO4[7] on P8_7 - Hitex board: ULPI pin 6

    // Configure the above inputs as GPIOs - even if this is already the reset state
    scu_pinmux (8, 5, MD_PDN, FUNC0);   // P8_5:  gpio 
    scu_pinmux (8, 4, MD_PDN, FUNC0);   // P8_4:  gpio 
    scu_pinmux (8, 7, MD_PDN, FUNC0);   // P8_7:  gpio 
}


/******************************************************************************
 * SCT interrupt handler
 * 
 *
 * Not used in this application 
 ******************************************************************************/
void SCT_IRQHandler (void)
{
    /* Acknowledge the interrupt souce */
    ;
}




/******************************************************************************
 * get_index
 * 
 *
 *  
 ******************************************************************************/
unsigned int get_index(unsigned char H1, unsigned char H2, unsigned char H3)
{
    if(H1 == 1)
        {
         if(H2 == 1) { return 1; }
         else if(H3 == 1) { return 5;}
         else return 0;
        }
        
        
    if(H2 == 1)
        {
        if(H3 == 1) { return 3; }
        else { return 2; }
        }
    
    return 4;
}


/******************************************************************************
 * set_Hall
 * 
 *
 *  
 ******************************************************************************/
 void set_Hall(unsigned char which)
{
    switch(which)
    {
        case 3:
                    LPC_GPIO4->SET = (1<<7);        // gpio 4_7
                    break;

        case 2:
                    LPC_GPIO4->SET = (1<<4);        // gpio 4_4
        break;

        case 1:
                    LPC_GPIO4->SET = (1<<5);    // gpio 4_5
        break;
    
    }
}



/******************************************************************************
 * clear_Hall
 * 
 *
 *  
 ******************************************************************************/
void clear_Hall(unsigned char which)
{
    switch(which)
    {
        case 3:
                    LPC_GPIO4->CLR = (1<<7);        // gpio 4_7
        break;

        case 2:
                    LPC_GPIO4->CLR = (1<<4);        // gpio 4_4
        break;

        case 1:
                    LPC_GPIO4->CLR = (1<<5);    // gpio 4_5
        break;
    
    }
}
    
    


/******************************************************************************
 * main
 * 
 *
 *  
 ******************************************************************************/
int main (void) 
{
    uint8_t index;
    uint32_t volatile i;
    uint8_t sensor_H1, sensor_H2, sensor_H3;
    
    // Set the Vector table offset if the program code is in external memory
    #ifdef EXTFLASH
        extern unsigned long __Vectors;
        SCB->VTOR = (unsigned long)&__Vectors;
    #endif
    
    // System and board specific initializations 
    SystemInit();

    // Enable clocks and PLL
    LPC_CREG->CREG0 |= (1<<1)|(1<<0);               // Enable 32 kHz & 1 kHz on osc32k 
    SetPL160M(SRC_IRC, 10);                         // Set PL160M @ 10*12 = 120 MHz with the IRC as source clock 
    SetClock(BASE_M3_CLK, SRC_PL160M_0, DIV1);      // Run base M3 clock from PL160M, div by 1 = no division 

    // Initialise the IO's needed for this example
    IOInit();
    SpecialIOInit();

    // Configure the EMC in case the software runs from external flash 
    #ifdef EXTFLASH
        EMC_Config_Pinmux();
        EMC_Config_Static();
    #endif



    // Global configuration of the SCT 
    LPC_CCU1->CLK_M3_SCT_CFG = 0x0001;              // Enable SCT branch clock in CCU1 
                                                    // RUN=1, AUTO=0, WAKE=0 
    LPC_SCT->CONFIG = 0x00000000;                   // Two separate 16-bit timers, clocked internally 
    LPC_SCT->CTRL_L = 0x001C + ((100-1) << 5);      // L counter: 
                                                    // Set prescaler = 1. Clear counter. 
                                                    // Keep in HALT. Prescaler = 1 
                                                    // Bidirectional 
    
    LPC_SCT->CTRL_H = 0x001C + ((100-1) << 5);      // H counter: 
                                                    // Set prescaler = 1. Clear counter. 
                                                    // Keep in HALT. Prescaler = 1 
                                                    // Bidirectional 

    // Now use the FSM code to configure the state machine 
    sct_fsm_init();

    // Other manual configuration and overrides 
    // Conflict resolution: CLR override for all 
    //    LPC_SCT->RES = (2 << (2*0)) | (2 << (2*1)) | (2 << (2*2)) | (2 << (2*3));


    // Bidirectional Control Register:
    // AH,AL,BH,BL,CH,CL: All are reversed when L counts down. (MAT0...5)
    LPC_SCT->BIDIRCTRL = (LPC_SCT->BIDIRCTRL & ~0x00000FFF) | 0x00000555;


    // Force start from first sector for test
    set_Hall(1);
    clear_Hall(2);
    clear_Hall(3);

    // Read from the inputs directly
    sensor_H1 =  (LPC_SCT->INPUT & (1<<4)) ? 1 : 0;
    sensor_H2 =  (LPC_SCT->INPUT & (1<<5))  ? 1 : 0;
    sensor_H3 = (LPC_SCT->INPUT & (1<<6))  ? 1 : 0;

    // Determine what the current configuration is
    index = get_index(sensor_H1, sensor_H2, sensor_H3);
    
    
    // Pre-program the state machine index for the current hall config
    LPC_SCT->STATE_L = position_table[index];  

    // Start the SCT 
    LPC_SCT->CTRL_U &= ~((1 << 2));  // Run L 
    


    //NVIC_EnableIRQ (UT_IRQn);

    while (1) 
    {
        // Could update the sensor positions with a periodic function, writing to the GPIO port
        // corresponding to the input channels H1, H2, H3
        // change the outputs in a row
        
        // The ugly delay loop
        for (i = 0; i < 1000000; i++);
                                                
        #ifdef FORWARD      
            // Update index modulo 6
            index = (index+1) % 6;
        #endif

        #ifdef BACKWARD
            (index-1) < 0 ? index = 5 : index--;        
        #endif

        // Simulate next hall step
        hall_config[index].H1 ? set_Hall(1) : clear_Hall(1);
        hall_config[index].H2 ? set_Hall(2) : clear_Hall(2);
        hall_config[index].H3 ? set_Hall(3) : clear_Hall(3);

        }
}


